/* Copyright (c) 2010 Nordic Semiconductor. All Rights Reserved.
 *
 * The information contained herein is property of Nordic Semiconductor ASA.
 * Terms and conditions of usage are described in detail in NORDIC
 * SEMICONDUCTOR STANDARD SOFTWARE LICENSE AGREEMENT. 
 *
 * Licensees are granted free, non-transferable use of the information. NO
 * WARRANTY of ANY KIND is provided. This heading must NOT be removed from
 * the file.
 */ 

/** @file
 * @brief Implementation of lib_ir_common
 */

#ifndef LIB_IR_COMMON_H__
#define LIB_IR_COMMON_H__

#include <stdint.h>
#include <stdbool.h>
           
#define NONE 0x00

/** The names of the different IR protocols 
 */
typedef enum
{
  PROTOCOL_NONE,
  PROTOCOL_NEC,
  PROTOCOL_RC5,
  PROTOCOL_SIRC_12BIT,
  PROTOCOL_SIRC_15BIT,
  PROTOCOL_SIRC_20BIT,
  PROTOCOL_UNKNOWN = 0xFF
} lib_ir_protocol_name_t;

/** The type of encoding of the IR protocol
 */
typedef enum
{
  ENC_PDE,      // Pulse Distance Encoding
  ENC_PLE,      // Pulse Length Encoding
  ENC_ME        // Manchester Encoding
} lib_ir_protocol_enc_t;

/** Flags for different parts of the IR sequence
 */
typedef enum
{
  HEADER  = 0x01,
  LEAD    = 0x02,
  DATA    = 0x04,
  REPEAT  = 0x08,
  TRAIL   = 0x10
} lib_ir_flags_t;

/** Structure to define the IR protocol
 */
typedef struct
{
  lib_ir_protocol_name_t  protocol;       // The name of the protocol
  uint8_t                 bits;           // The number of bits in the sequence
  uint8_t                 toggle_bit;     // The position of the toggle bit if any
  uint8_t                 repeat_bit;     // The position of the repeat bit if any
  lib_ir_protocol_enc_t   encoding;       // The type of encoding the protocol has
  uint16_t                frequency;      // The frequency of the IR pulses
  uint8_t                 duty_cycle;     // The duty cycle of the IR pulses
  uint8_t                 min_repeat;     // The minimum number of repeated sequences
  uint8_t                 flags;          // The flags defining the parts of the protocol
  int16_t                 header[2];      // The time in s of a header pulse sequence (Time > 0 => Pulse, Time < 0 => Space)
  int16_t                 lead;           // The time in s of a lead sequence (Time > 0 => Pulse, Time < 0 => Space)
  int16_t                 one[2];         // The time in s of a logical one sequence (Time > 0 => Pulse, Time < 0 => Space)
  int16_t                 zero[2];        // The time in s of a logical zero sequence (Time > 0 => Pulse, Time < 0 => Space)
  int16_t                 trail;          // The time in s of a trail sequence (Time > 0 => Pulse, Time < 0 => Space)
  int16_t                 repeat[2];      // The time in s of a lead sequence (Time > 0 => Pulse, Time < 0 => Space)
  uint32_t                gap;            // The time in s of between repeated sequences
} lib_ir_protocol_struct_t;

/** Definition of the NEC protocol
 */
static code lib_ir_protocol_struct_t lib_ir_protocol_nec = {
  PROTOCOL_NEC,                   // name 
  32,                             // bits
  NONE,                           // toggle_bit
  NONE,                           // repeat_bit
  ENC_PDE,                        // encoding
  38000,                          // frequency   
  33,                             // duty_cycle  
  NONE,                           // min_repeat
  HEADER | DATA | TRAIL,          // flags
  9000, -4500,                    // header       
  NONE,                           // lead
  563, -1688,                     // one          
  563, -563,                      // zero         
  563,                            // trail       
  9000, -2250,                    // repeat       
  108000,                         // gap 
};

/** Definition of the RC-5 protocol
 */
static code lib_ir_protocol_struct_t lib_ir_protocol_rc5 = {
  PROTOCOL_RC5, // name 
  13,           // bits          
  1,            // toggle_bit     
  NONE,         // repeat_bit
  ENC_ME,       // encoding
  36000,        // frequency (35750)
  25,           // duty_cycle  
  NONE,         // min_repeat
  LEAD | DATA,  // flags
  NONE, NONE,   // header
  889,          // lead        
  -889, 889,    // one          
  889, -889,    // zero         
  NONE,         // trail       
  NONE, NONE,   // repeat
  114000,       // gap         
}; 

/** Definition of the SIRC 12 bit protocol
 */
static code lib_ir_protocol_struct_t lib_ir_protocol_sirc_12bit = {
  PROTOCOL_SIRC_12BIT,  // protocol
  12,                   // bits           
  NONE,                 // toggle_bit
  NONE,                 // repeat_bit
  ENC_PLE,              // encoding
  40000,                // frequency   
  33,                   // duty_cycle  
  2,                    // min_repeat
  HEADER | DATA,        // flags     
  2400, -600,           // header      
  NONE,                 // lead
  1200, -600,           // one         
  600, -600,            // zero         
  NONE,                 // trail       
  NONE, NONE,           // repeat
  45000,                // gap         
};

/** Definition of the SIRC 15 bit protocol
 */
static code lib_ir_protocol_struct_t lib_ir_protocol_sirc_15bit = {
  PROTOCOL_SIRC_15BIT,  // protocol
  15,                   // bits           
  NONE,                 // toggle_bit
  NONE,                 // repeat_bit
  ENC_PLE,              // encoding
  40000,                // frequency   
  33,                   // duty_cycle
  2,                    // min_repeat  
  HEADER | DATA,        // flags   
  2400, -600,           // header      
  NONE,                 // lead
  1200, -600,           // one         
  600, -600,            // zero         
  NONE,                 // trail
  NONE, NONE,           // repeat
  45000,                // gap 
};

/** Definition of the SIRC 20 bit protocol
 */
static code lib_ir_protocol_struct_t lib_ir_protocol_sirc_20bit = {
  PROTOCOL_SIRC_20BIT,  // protocol
  20,                   // bits           
  NONE,                 // toggle_bit
  NONE,                 // repeat_bit
  ENC_PLE,              // encoding
  40000,                // frequency   
  33,                   // duty_cycle
  2,                    // min_repeat  
  HEADER | DATA,        // flags   
  2400, -600,           // header      
  NONE,                 // lead
  1200, -600,           // one         
  600, -600,            // zero         
  NONE,                 // trail
  NONE, NONE,           // repeat
  45000,                // gap 
};

#endif
